import { useRef, useEffect } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

interface QuoteBannerProps {
  image: string;
  quote: string;
  subline?: string;
  cta?: string;
  ctaHref?: string;
  overlayOpacity?: number;
}

const QuoteBanner = ({
  image,
  quote,
  subline,
  cta,
  ctaHref = '#',
  overlayOpacity = 0.45,
}: QuoteBannerProps) => {
  const sectionRef = useRef<HTMLElement>(null);
  const bgRef = useRef<HTMLDivElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);
  const quoteRef = useRef<HTMLHeadingElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      const scrollTl = gsap.timeline({
        scrollTrigger: {
          trigger: sectionRef.current,
          start: 'top top',
          end: '+=125%',
          pin: true,
          scrub: 0.6,
        },
      });

      // Background animation
      scrollTl.fromTo(
        bgRef.current,
        { scale: 1.1, y: '8vh' },
        { scale: 1, y: 0, ease: 'none' },
        0
      );
      scrollTl.to(
        bgRef.current,
        { scale: 1.06, y: '-6vh', ease: 'none' },
        0.7
      );

      // Quote text animation
      if (quoteRef.current) {
        const words = quoteRef.current.querySelectorAll('.word');
        scrollTl.fromTo(
          words,
          { opacity: 0, y: 40 },
          { opacity: 1, y: 0, stagger: 0.02, ease: 'none' },
          0.05
        );
        scrollTl.to(
          contentRef.current,
          { y: '-16vh', opacity: 0, ease: 'power2.in' },
          0.7
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section
      ref={sectionRef}
      className="relative w-full h-screen overflow-hidden z-30"
    >
      {/* Background Image */}
      <div ref={bgRef} className="absolute inset-0 w-full h-full">
        <img
          src={image}
          alt=""
          className="w-full h-full object-cover"
        />
        {/* Overlay */}
        <div
          className="absolute inset-0 transition-opacity duration-300"
          style={{ backgroundColor: `rgba(16, 20, 27, ${overlayOpacity})` }}
        />
      </div>

      {/* Content */}
      <div
        ref={contentRef}
        className="relative z-10 flex flex-col items-center justify-center h-full px-6 text-center"
      >
        <h2
          ref={quoteRef}
          className="text-white text-3xl sm:text-4xl md:text-5xl lg:text-6xl xl:text-7xl font-black uppercase tracking-tight max-w-4xl mb-6"
        >
          {quote.split(' ').map((word, i) => (
            <span key={i} className="word inline-block mr-3">
              {word}
            </span>
          ))}
        </h2>

        {subline && (
          <p className="text-white/70 text-base sm:text-lg md:text-xl max-w-xl mb-8 font-light">
            {subline}
          </p>
        )}

        {cta && (
          <a
            href={ctaHref}
            className="text-[#D4A27F] text-sm font-medium underline underline-offset-4 hover:text-white transition-colors duration-300"
          >
            {cta}
          </a>
        )}
      </div>
    </section>
  );
};

export default QuoteBanner;
