import { useRef, useEffect } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { MapPin, ArrowRight, Globe } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

const locations = [
  { name: 'Sedona, AZ', coords: { x: 18, y: 45 } },
  { name: 'Reykjavík, IS', coords: { x: 42, y: 22 } },
  { name: 'Cape Town, ZA', coords: { x: 52, y: 78 } },
  { name: 'Tokyo, JP', coords: { x: 82, y: 38 } },
  { name: 'Maui, HI', coords: { x: 8, y: 48 } },
  { name: 'Maui, HI', coords: { x: 75, y: 55 } },
];

const MapTeaser = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const mapRef = useRef<HTMLDivElement>(null);
  const sidebarRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Map entrance
      gsap.fromTo(
        mapRef.current,
        { opacity: 0, x: '-8vw', scale: 0.98 },
        {
          opacity: 1,
          x: 0,
          scale: 1,
          duration: 0.8,
          scrollTrigger: {
            trigger: sectionRef.current,
            start: 'top 80%',
            end: 'top 55%',
            scrub: 1,
          },
        }
      );

      // Sidebar entrance
      gsap.fromTo(
        sidebarRef.current,
        { opacity: 0, x: '6vw' },
        {
          opacity: 1,
          x: 0,
          duration: 0.8,
          scrollTrigger: {
            trigger: sectionRef.current,
            start: 'top 80%',
            end: 'top 55%',
            scrub: 1,
          },
        }
      );

      // Pins stagger animation
      const pins = mapRef.current?.querySelectorAll('.map-pin');
      if (pins) {
        gsap.fromTo(
          pins,
          { opacity: 0, y: 12, scale: 0.8 },
          {
            opacity: 1,
            y: 0,
            scale: 1,
            duration: 0.4,
            stagger: 0.06,
            scrollTrigger: {
              trigger: mapRef.current,
              start: 'top 70%',
              toggleActions: 'play none none reverse',
            },
          }
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section
      id="map"
      ref={sectionRef}
      className="relative w-full py-20 lg:py-32 bg-[#F6F4EF] z-20"
    >
      <div className="w-full px-6 lg:px-12">
        <div className="flex flex-col lg:flex-row gap-8 lg:gap-12">
          {/* Map Panel */}
          <div
            ref={mapRef}
            className="relative w-full lg:w-3/5 aspect-[4/3] lg:aspect-[16/10] bg-[#E8E4DC] rounded-xl overflow-hidden"
          >
            {/* Stylized world map background */}
            <div className="absolute inset-0 opacity-30">
              <svg
                viewBox="0 0 100 60"
                className="w-full h-full"
                preserveAspectRatio="xMidYMid slice"
              >
                {/* Simplified world map paths */}
                <path
                  d="M20,15 Q25,10 30,15 T40,18 Q45,15 50,20 T60,18 Q65,12 70,18 T80,15 Q85,20 80,25 T75,30 Q80,35 75,40 T70,45 Q65,50 60,45 T50,48 Q45,52 40,48 T30,50 Q25,45 20,50 T15,45 Q10,40 15,35 T20,30 Q15,25 20,20 T20,15"
                  fill="#C4B8A8"
                />
                <path
                  d="M45,25 Q50,22 55,25 T60,30 Q55,35 50,32 T45,35 Q40,32 45,25"
                  fill="#C4B8A8"
                />
                <path
                  d="M75,35 Q80,32 85,35 T88,42 Q83,45 78,42 T75,35"
                  fill="#C4B8A8"
                />
              </svg>
            </div>

            {/* Grid lines */}
            <div className="absolute inset-0 opacity-10">
              <div className="w-full h-full" style={{
                backgroundImage: `
                  linear-gradient(to right, #1A1D23 1px, transparent 1px),
                  linear-gradient(to bottom, #1A1D23 1px, transparent 1px)
                `,
                backgroundSize: '20% 20%'
              }} />
            </div>

            {/* Location Pins */}
            {locations.map((location, index) => (
              <div
                key={index}
                className="map-pin absolute group cursor-pointer"
                style={{
                  left: `${location.coords.x}%`,
                  top: `${location.coords.y}%`,
                }}
              >
                <div className="relative">
                  <div className="w-4 h-4 bg-[#D4A27F] rounded-full border-2 border-white shadow-lg transition-transform duration-300 group-hover:scale-125" />
                  <div className="absolute inset-0 w-4 h-4 bg-[#D4A27F] rounded-full animate-ping opacity-40" />
                  
                  {/* Tooltip */}
                  <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-3 py-1.5 bg-[#1A1D23] text-white text-xs font-medium rounded-lg opacity-0 group-hover:opacity-100 transition-opacity duration-300 whitespace-nowrap pointer-events-none">
                    {location.name}
                    <div className="absolute top-full left-1/2 -translate-x-1/2 border-4 border-transparent border-t-[#1A1D23]" />
                  </div>
                </div>
              </div>
            ))}

            {/* Map overlay gradient */}
            <div className="absolute inset-0 bg-gradient-to-t from-[#F6F4EF]/20 to-transparent pointer-events-none" />
          </div>

          {/* Sidebar */}
          <div
            ref={sidebarRef}
            className="w-full lg:w-2/5 flex flex-col justify-center"
          >
            <div className="flex items-center gap-2 mb-4">
              <Globe className="w-5 h-5 text-[#D4A27F]" />
              <span className="section-label">Explore</span>
            </div>
            
            <h2 className="text-3xl sm:text-4xl lg:text-5xl font-black text-[#1A1D23] uppercase tracking-tight mb-6">
              Explore Sunrises Around the World
            </h2>
            
            <p className="text-[#6E727A] text-base lg:text-lg mb-8 leading-relaxed">
              Click a pin to see what 5AM looks like there—airports, trails, 
              rooftops, and quiet streets. Every location has its own quiet magic.
            </p>

            {/* Location list */}
            <div className="space-y-3 mb-8">
              {['Sedona, AZ', 'Reykjavík, IS', 'Cape Town, ZA', 'Tokyo, JP', 'Maui, HI'].map((location) => (
                <button
                  key={location}
                  className="w-full flex items-center justify-between p-3 rounded-lg bg-white border border-[#1A1D23]/5 hover:bg-[#1A1D23]/5 hover:border-[#D4A27F]/30 transition-all duration-300 group"
                >
                  <div className="flex items-center gap-3">
                    <MapPin className="w-4 h-4 text-[#D4A27F]" />
                    <span className="text-[#1A1D23] font-medium">{location}</span>
                  </div>
                  <ArrowRight className="w-4 h-4 text-[#6E727A] group-hover:text-[#D4A27F] group-hover:translate-x-1 transition-all duration-300" />
                </button>
              ))}
            </div>

            <button className="btn-primary self-start">
              Open Sunrise Map
            </button>
          </div>
        </div>
      </div>
    </section>
  );
};

export default MapTeaser;
