import { useRef, useEffect, useState } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { MapPin, Clock, Sun, Plane, Mountain, Coffee, Building, Heart } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

interface Post {
  id: number;
  image?: string;
  title: string;
  location: string;
  time: string;
  category: 'sunrise' | 'travel' | 'reflection' | 'city' | 'hiking' | 'coffee';
  excerpt?: string;
  likes: number;
}

const posts: Post[] = [
  {
    id: 1,
    image: '/grid_foggy_river.jpg',
    title: 'Fog over the river. No filters.',
    location: 'Portland, OR',
    time: '5:28 AM',
    category: 'sunrise',
    likes: 234,
  },
  {
    id: 2,
    image: '/grid_tokyo_train.jpg',
    title: 'Early train, empty streets.',
    location: 'Tokyo, JP',
    time: '5:02 AM',
    category: 'travel',
    likes: 189,
  },
  {
    id: 3,
    image: '/grid_coffee_shop.jpg',
    title: 'Coffee shop lights on an empty corner.',
    location: 'Brooklyn, NY',
    time: '5:09 AM',
    category: 'coffee',
    likes: 312,
  },
  {
    id: 4,
    image: '/grid_pine_forest.jpg',
    title: 'First light through the pines.',
    location: 'Banff, CA',
    time: '5:33 AM',
    category: 'hiking',
    likes: 456,
  },
  {
    id: 5,
    image: '/grid_foggy_harbor.jpg',
    title: 'Foggy Harbor Pause',
    location: 'Seattle, WA',
    time: '5:21 AM',
    category: 'sunrise',
    likes: 278,
  },
  {
    id: 6,
    image: '/grid_desert_road.jpg',
    title: 'Desert Road First Light',
    location: 'Joshua Tree, CA',
    time: '5:44 AM',
    category: 'travel',
    likes: 367,
  },
  {
    id: 7,
    image: '/grid_rooftop.jpg',
    title: 'Rooftop Waiting',
    location: 'Mexico City, MX',
    time: '5:56 AM',
    category: 'city',
    likes: 198,
  },
  {
    id: 8,
    image: '/grid_lake_dawn.jpg',
    title: 'Lake Before the Breeze',
    location: 'Queenstown, NZ',
    time: '5:38 AM',
    category: 'sunrise',
    likes: 523,
  },
  {
    id: 9,
    image: '/grid_alley_cats.jpg',
    title: 'Alley Cats & Streetlights',
    location: 'Istanbul, TR',
    time: '5:12 AM',
    category: 'city',
    likes: 289,
  },
];

const categoryIcons = {
  sunrise: Sun,
  travel: Plane,
  reflection: Coffee,
  city: Building,
  hiking: Mountain,
  coffee: Coffee,
};

const categoryLabels = {
  sunrise: 'Sunrise',
  travel: 'Travel',
  reflection: 'Reflection',
  city: 'City',
  hiking: 'Hiking',
  coffee: 'Coffee',
};

const LatestGrid = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const headerRef = useRef<HTMLDivElement>(null);
  const gridRef = useRef<HTMLDivElement>(null);
  const [activeFilter, setActiveFilter] = useState<string>('all');

  const filters = ['all', 'sunrise', 'travel', 'reflection', 'city', 'hiking'];

  const filteredPosts = activeFilter === 'all' 
    ? posts 
    : posts.filter(post => post.category === activeFilter);

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Header animation
      gsap.fromTo(
        headerRef.current,
        { opacity: 0, y: 30 },
        {
          opacity: 1,
          y: 0,
          duration: 0.6,
          scrollTrigger: {
            trigger: headerRef.current,
            start: 'top 80%',
            end: 'top 60%',
            scrub: 1,
          },
        }
      );

      // Cards staggered animation
      const cards = gridRef.current?.querySelectorAll('.post-card');
      if (cards) {
        gsap.fromTo(
          cards,
          { opacity: 0, y: 60, scale: 0.98 },
          {
            opacity: 1,
            y: 0,
            scale: 1,
            duration: 0.6,
            stagger: 0.08,
            scrollTrigger: {
              trigger: gridRef.current,
              start: 'top 80%',
              end: 'top 40%',
              scrub: 1,
            },
          }
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, [filteredPosts]);

  return (
    <section
      id="stories"
      ref={sectionRef}
      className="relative w-full py-20 lg:py-32 bg-[#F6F4EF] z-20"
    >
      <div className="w-full px-6 lg:px-12">
        {/* Header */}
        <div ref={headerRef} className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6 mb-10 lg:mb-14">
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-black text-[#1A1D23] uppercase tracking-tight">
            Latest 5AM Moments
          </h2>
          
          {/* Filters */}
          <div className="flex flex-wrap gap-2">
            {filters.map((filter) => (
              <button
                key={filter}
                onClick={() => setActiveFilter(filter)}
                className={`px-4 py-2 text-sm font-medium rounded-full transition-all duration-300 ${
                  activeFilter === filter
                    ? 'bg-[#1A1D23] text-white'
                    : 'bg-transparent text-[#6E727A] hover:bg-[#1A1D23]/5'
                }`}
              >
                {filter.charAt(0).toUpperCase() + filter.slice(1)}
              </button>
            ))}
          </div>
        </div>

        {/* Grid */}
        <div
          ref={gridRef}
          className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 lg:gap-8"
        >
          {filteredPosts.map((post) => {
            const Icon = categoryIcons[post.category];
            return (
              <article
                key={post.id}
                className="post-card group relative bg-white rounded-xl overflow-hidden border border-[#1A1D23]/5 transition-all duration-300 hover:-translate-y-1.5 hover:card-shadow-hover"
              >
                {/* Image */}
                <div className="relative aspect-[4/3] overflow-hidden">
                  <img
                    src={post.image}
                    alt={post.title}
                    className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                  />
                  {/* Category badge */}
                  <div className="absolute top-3 left-3 flex items-center gap-1.5 px-2.5 py-1 bg-white/90 backdrop-blur-sm rounded-full">
                    <Icon className="w-3.5 h-3.5 text-[#D4A27F]" />
                    <span className="text-xs font-medium text-[#1A1D23]">
                      {categoryLabels[post.category]}
                    </span>
                  </div>
                </div>

                {/* Content */}
                <div className="p-5">
                  <h3 className="text-lg font-semibold text-[#1A1D23] mb-3 line-clamp-2 group-hover:text-[#D4A27F] transition-colors duration-300">
                    {post.title}
                  </h3>
                  
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3 text-xs text-[#6E727A]">
                      <span className="flex items-center gap-1">
                        <MapPin className="w-3.5 h-3.5" />
                        {post.location}
                      </span>
                      <span className="flex items-center gap-1">
                        <Clock className="w-3.5 h-3.5" />
                        {post.time}
                      </span>
                    </div>
                    
                    <div className="flex items-center gap-1 text-xs text-[#6E727A]">
                      <Heart className="w-3.5 h-3.5" />
                      {post.likes}
                    </div>
                  </div>
                </div>
              </article>
            );
          })}
        </div>

        {/* View All Button */}
        <div className="flex justify-center mt-12">
          <button className="btn-secondary">
            View All Posts
          </button>
        </div>
      </div>
    </section>
  );
};

export default LatestGrid;
