import { useRef, useEffect } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { MapPin, Clock, Heart, ArrowRight } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

interface FavoritePost {
  id: number;
  image: string;
  title: string;
  location: string;
  time: string;
  author: string;
  likes: number;
}

const favorites: FavoritePost[] = [
  {
    id: 1,
    image: '/grid_foggy_harbor.jpg',
    title: 'Foggy Harbor Pause',
    location: 'Seattle, WA',
    time: '5:21 AM',
    author: 'Lena',
    likes: 892,
  },
  {
    id: 2,
    image: '/grid_desert_road.jpg',
    title: 'Desert Road First Light',
    location: 'Joshua Tree, CA',
    time: '5:44 AM',
    author: 'Marcus',
    likes: 756,
  },
  {
    id: 3,
    image: '/grid_rooftop.jpg',
    title: 'Rooftop Waiting',
    location: 'Mexico City, MX',
    time: '5:56 AM',
    author: 'Camila',
    likes: 634,
  },
  {
    id: 4,
    image: '/grid_lake_dawn.jpg',
    title: 'Lake Before the Breeze',
    location: 'Queenstown, NZ',
    time: '5:38 AM',
    author: 'Tom',
    likes: 1247,
  },
  {
    id: 5,
    image: '/grid_alley_cats.jpg',
    title: 'Alley Cats & Streetlights',
    location: 'Istanbul, TR',
    time: '5:12 AM',
    author: 'Selin',
    likes: 523,
  },
  {
    id: 6,
    image: '/grid_summit_thermos.jpg',
    title: 'Summit Thermos Moment',
    location: 'Chamonix, FR',
    time: '5:29 AM',
    author: 'Jules',
    likes: 1089,
  },
];

const CommunityFavorites = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const headerRef = useRef<HTMLDivElement>(null);
  const gridRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Header animation
      gsap.fromTo(
        headerRef.current,
        { opacity: 0, y: 24 },
        {
          opacity: 1,
          y: 0,
          duration: 0.6,
          scrollTrigger: {
            trigger: headerRef.current,
            start: 'top 80%',
            end: 'top 60%',
            scrub: 1,
          },
        }
      );

      // Cards staggered animation
      const cards = gridRef.current?.querySelectorAll('.favorite-card');
      if (cards) {
        gsap.fromTo(
          cards,
          { opacity: 0, y: 50 },
          {
            opacity: 1,
            y: 0,
            duration: 0.6,
            stagger: 0.08,
            scrollTrigger: {
              trigger: gridRef.current,
              start: 'top 80%',
              end: 'top 40%',
              scrub: 1,
            },
          }
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section
      id="travel"
      ref={sectionRef}
      className="relative w-full py-20 lg:py-32 bg-[#F6F4EF] z-20"
    >
      <div className="w-full px-6 lg:px-12">
        {/* Header */}
        <div ref={headerRef} className="flex items-center justify-between mb-10 lg:mb-14">
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-black text-[#1A1D23] uppercase tracking-tight">
            Community Favorites
          </h2>
          <a
            href="#"
            className="hidden sm:flex items-center gap-2 text-[#6E727A] hover:text-[#D4A27F] transition-colors duration-300 font-medium"
          >
            View all
            <ArrowRight className="w-4 h-4" />
          </a>
        </div>

        {/* Grid */}
        <div
          ref={gridRef}
          className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 lg:gap-8"
        >
          {favorites.map((post) => (
            <article
              key={post.id}
              className="favorite-card group relative bg-white rounded-xl overflow-hidden border border-[#1A1D23]/5 transition-all duration-300 hover:-translate-y-1.5 hover:card-shadow-hover"
            >
              {/* Image */}
              <div className="relative aspect-[4/3] overflow-hidden">
                <img
                  src={post.image}
                  alt={post.title}
                  className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                />
              </div>

              {/* Content */}
              <div className="p-5">
                <h3 className="text-lg font-semibold text-[#1A1D23] mb-2 line-clamp-1 group-hover:text-[#D4A27F] transition-colors duration-300">
                  {post.title}
                </h3>
                
                <div className="flex items-center gap-3 text-xs text-[#6E727A] mb-3">
                  <span className="flex items-center gap-1">
                    <MapPin className="w-3.5 h-3.5" />
                    {post.location}
                  </span>
                  <span className="flex items-center gap-1">
                    <Clock className="w-3.5 h-3.5" />
                    {post.time}
                  </span>
                </div>

                <div className="flex items-center justify-between pt-3 border-t border-[#1A1D23]/5">
                  <span className="text-sm text-[#6E727A]">
                    By <span className="text-[#1A1D23] font-medium">{post.author}</span>
                  </span>
                  <div className="flex items-center gap-1 text-sm text-[#6E727A]">
                    <Heart className="w-4 h-4 text-[#D4A27F]" />
                    {post.likes}
                  </div>
                </div>
              </div>
            </article>
          ))}
        </div>

        {/* Mobile View All */}
        <div className="flex justify-center mt-8 sm:hidden">
          <a
            href="#"
            className="flex items-center gap-2 text-[#6E727A] hover:text-[#D4A27F] transition-colors duration-300 font-medium"
          >
            View all
            <ArrowRight className="w-4 h-4" />
          </a>
        </div>
      </div>
    </section>
  );
};

export default CommunityFavorites;
