import { useEffect } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

import Header from './sections/Header';
import Hero from './sections/Hero';
import FeaturedSlider from './sections/FeaturedSlider';
import LatestGrid from './sections/LatestGrid';
import QuoteBanner from './sections/QuoteBanner';
import MapTeaser from './sections/MapTeaser';
import CommunityFavorites from './sections/CommunityFavorites';
import SubmitForm from './sections/SubmitForm';
import Footer from './sections/Footer';

gsap.registerPlugin(ScrollTrigger);

interface PinnedRange {
  start: number;
  end: number;
  center: number;
}

function App() {
  useEffect(() => {
    // Wait for all ScrollTriggers to be created
    const timeout = setTimeout(() => {
      const pinned = ScrollTrigger.getAll()
        .filter((st: ScrollTrigger) => st.vars.pin)
        .sort((a: ScrollTrigger, b: ScrollTrigger) => a.start - b.start);
      
      const maxScroll = ScrollTrigger.maxScroll(window);
      
      if (!maxScroll || pinned.length === 0) return;

      // Build ranges and snap targets from pinned sections
      const pinnedRanges: PinnedRange[] = pinned.map((st: ScrollTrigger) => ({
        start: st.start / maxScroll,
        end: (st.end ?? st.start) / maxScroll,
        center: (st.start + ((st.end ?? st.start) - st.start) * 0.5) / maxScroll,
      }));

      // Global snap configuration
      ScrollTrigger.create({
        snap: {
          snapTo: (value: number) => {
            // Check if within any pinned range (with buffer)
            const inPinned = pinnedRanges.some(
              (r: PinnedRange) => value >= r.start - 0.02 && value <= r.end + 0.02
            );
            
            // If not in a pinned section, allow free scroll
            if (!inPinned) return value;

            // Find nearest pinned center
            const target = pinnedRanges.reduce(
              (closest: number, r: PinnedRange) =>
                Math.abs(r.center - value) < Math.abs(closest - value)
                  ? r.center
                  : closest,
              pinnedRanges[0]?.center ?? 0
            );

            return target;
          },
          duration: { min: 0.15, max: 0.35 },
          delay: 0,
          ease: 'power2.out',
        },
      });
    }, 100);

    return () => {
      clearTimeout(timeout);
      ScrollTrigger.getAll().forEach((st: ScrollTrigger) => st.kill());
    };
  }, []);

  return (
    <div className="relative bg-[#F6F4EF]">
      {/* Noise overlay */}
      <div className="noise-overlay" />
      
      {/* Header */}
      <Header />

      {/* Main content */}
      <main className="relative">
        {/* Section 1: Hero - pin: true */}
        <Hero />

        {/* Section 2: Featured Slider - pin: false */}
        <FeaturedSlider />

        {/* Section 3: Latest Grid - pin: false */}
        <LatestGrid />

        {/* Section 4: Quote Banner 1 - pin: true */}
        <QuoteBanner
          image="/quote_dawn_landscape.jpg"
          quote="Some mornings change your life."
          subline="— a note from the 5AM journal"
          cta="Read the journal"
          ctaHref="#stories"
        />

        {/* Section 5: Map Teaser - pin: false */}
        <MapTeaser />

        {/* Section 6: Quote Banner 2 - pin: true */}
        <QuoteBanner
          image="/quote_mountain_ridge.jpg"
          quote="It's quiet up here."
          subline="If you've ever hiked in the dark to see the light, you already belong here."
          cta="Share your trail"
          ctaHref="#submit"
        />

        {/* Section 7: Community Favorites - pin: false */}
        <CommunityFavorites />

        {/* Section 8: Quote Banner 3 - pin: true */}
        <QuoteBanner
          image="/quote_ocean_horizon.jpg"
          quote="The world is still asleep."
          subline="Post your sunrise. Read someone else's. Stay awake together."
          cta="Join the 5AM Journal"
          ctaHref="#submit"
        />

        {/* Section 9: Submit Form - pin: false */}
        <SubmitForm />

        {/* Section 10: Footer - pin: true */}
        <Footer />
      </main>
    </div>
  );
}

export default App;
